/*
 * udma_task
 *
 * Copyright (C) 2022 Texas Instruments Incorporated
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

/******************************************************************************
 *
 * The vuDMATask task first initializes the demonstration with initial messages
 * over the UART terminal and pre-filling the buffer.  The CPU usage monitor is
 * also initialized.  Then the uDMA peripheral of the TM4C123GH6PM MCU is
 * configured for automatic mode using the software transfer channel.  This
 * will allow for the memory-to-memory DMA transfers to occur repeatedly.
 *
 * The prvMemTransfer task works in conjunction with a software time that is
 * configured to expire at a one second interval.  Upon timeout, the timer sends
 * a notification to the prvMemTransfer which then calculates the amount of
 * bytes transferred in one second and outputs the result over UART.  After
 * running for ten seconds, the task self-terminates.
 *
 * The processor is put to sleep when it is not doing anything, and this allows
 * collection of CPU usage data to see how much CPU is being used while the
 * data transfers are ongoing.
 *
 */

/* Standard includes. */
#include <stdio.h>
#include <stdint.h>
#include <stdbool.h>

/* Kernel includes. */
#include "FreeRTOS.h"
#include "task.h"
#include "semphr.h"

/* Hardware includes. */
#include "inc/hw_ints.h"
#include "inc/hw_memmap.h"
#include "driverlib/gpio.h"
#include "driverlib/interrupt.h"
#include "driverlib/sysctl.h"
#include "driverlib/rom.h"
#include "driverlib/rom_map.h"
#include "driverlib/udma.h"
#include "drivers/rtos_hw_drivers.h"
#include "utils/uartstdio.h"
#include "utils/cpu_usage.h"

/*-----------------------------------------------------------*/

/*
 * The number of ticks per second for the software timer.
 */
#define SYSTICKS_PER_SECOND     100

/*
 *  The size of the memory transfer source and destination buffers (in words).
 */
#define MEM_BUFFER_SIZE         1024

/*
 * The source and destination buffers used for memory transfers.
 */
static uint32_t g_ui32SrcBuf[MEM_BUFFER_SIZE];
static uint32_t g_ui32DstBuf[MEM_BUFFER_SIZE];

/*
 * The count of memory uDMA transfer blocks.  This value is incremented by the
 * uDMA interrupt handler whenever a memory block transfer is completed.
 */
static uint32_t g_ui32MemXferCount = 0;

/*
 * The control table used by the uDMA controller.  This table must be aligned
 * to a 1024 byte boundary.
 */
#if defined(ewarm)
#pragma data_alignment=1024
uint8_t ui8ControlTable[1024];
#elif defined(ccs)
#pragma DATA_ALIGN(ui8ControlTable, 1024)
uint8_t ui8ControlTable[1024];
#else
uint8_t ui8ControlTable[1024] __attribute__ ((aligned(1024)));
#endif

/*
 * The count of uDMA errors.  This value is incremented by the uDMA error
 * handler.
*/
static uint32_t g_ui32uDMAErrCount = 0;

/*
 * The count of times the uDMA interrupt occurred but the uDMA transfer was not
 * complete.  This should remain 0.
*/
static uint32_t g_ui32BadISR = 0;

/*
 * The CPU usage in percent, in 16.16 fixed point format.
 */
uint32_t g_ui32CPUUsage;

/*
 * The notification used by the task.
 */
TaskHandle_t xTaskuDMAHandle = NULL;

/*
 * The tasks as described in the comments at the top of this file.
 */
static void prvMemTransfer( void *pvParameters );

/*
 * Called by main() to create the simple blinky style application.
 */
void vuDMATask( void );

/*
 * Initialize various variables, status messages, and system features for the
 * application demonstration.
 */
static void prvInitApplication( void );

/*
 * Hardware configuration for the uDMA module for memory transfers.
 */
static void prvInitSWTransfer( void );
/*-----------------------------------------------------------*/

void vuDMATask( void )
{
    /* Initialize application parameters. */
    prvInitApplication();

    /* Initialize the uDMA memory to memory transfers. */
    prvInitSWTransfer();

    /* Create the task as described in the comments at the top of this file.
     *
     * The xTaskCreate parameters in order are:
     *  - The function that implements the task.
     *  - The text name for memory transfer task - for debug only as it is
     *    not used by the kernel.
     *  - The size of the stack to allocate to the task.
     *  - The parameter passed to the task - just to check the functionality.
     *  - The priority assigned to the task.
     *  - The task handle used for the task notify. */
    xTaskCreate( prvMemTransfer,
                 "MemTransfer",
                 configMINIMAL_STACK_SIZE,
                 NULL,
                 tskIDLE_PRIORITY + 1,
                 &xTaskuDMAHandle );
}
/*-----------------------------------------------------------*/

static void prvMemTransfer( void *pvParameters )
{
static uint32_t ui32PrevSeconds=0;
static uint32_t ui32PrevXferCount;
uint32_t ui32XfersCompleted;
uint32_t ui32BytesTransferred;

    while (1)
    {
        /* Block until the task notification is received. When
         * the notification is received, it means the SW timer
         * has expired */
        if (ulTaskNotifyTake(pdTRUE, portMAX_DELAY) == pdPASS)
        {
            /* Print a message to the display showing the CPU usage percent.
             * The fractional part of the percent value is ignored. */
            UARTprintf("\r%3d%%   ", g_ui32CPUUsage >> 16);

            /* Calculate how many memory transfers have occurred since the last
             * second. */
            ui32XfersCompleted = g_ui32MemXferCount - ui32PrevXferCount;

            /* Remember the new transfer count. */
            ui32PrevXferCount = g_ui32MemXferCount;

            /* Compute how many bytes were transferred in the memory transfer
             * since the last second. */
            ui32BytesTransferred = ui32XfersCompleted * MEM_BUFFER_SIZE * 4;

            /* Print a message showing the memory transfer rate. */
            if(ui32BytesTransferred >= 100000000)
            {
                UARTprintf("%3d MB/s   ", ui32BytesTransferred / 1000000);
            }
            else if(ui32BytesTransferred >= 10000000)
            {
                UARTprintf("%2d.%01d MB/s  ", ui32BytesTransferred / 1000000,
                           (ui32BytesTransferred % 1000000) / 100000);
            }
            else if(ui32BytesTransferred >= 1000000)
            {
                UARTprintf("%1d.%02d MB/s  ", ui32BytesTransferred / 1000000,
                           (ui32BytesTransferred % 1000000) / 10000);
            }
            else if(ui32BytesTransferred >= 100000)
            {
                UARTprintf("%3d KB/s   ", ui32BytesTransferred / 1000);
            }
            else if(ui32BytesTransferred >= 10000)
            {
                UARTprintf("%2d.%01d KB/s  ", ui32BytesTransferred / 1000,
                           (ui32BytesTransferred % 1000) / 100);
            }
            else if(ui32BytesTransferred >= 1000)
            {
                UARTprintf("%1d.%02d KB/s  ", ui32BytesTransferred / 1000,
                           (ui32BytesTransferred % 1000) / 10);
            }
            else if(ui32BytesTransferred >= 100)
            {
                UARTprintf("%3d B/s    ", ui32BytesTransferred);
            }
            else if(ui32BytesTransferred >= 10)
            {
                UARTprintf("%2d B/s     ", ui32BytesTransferred);
            }
            else
            {
                UARTprintf("%1d B/s      ", ui32BytesTransferred);
            }

            /* Print a remaining time to make it more apparent that there is
             * something happening. */
            UARTprintf("%2ds", 10 - ui32PrevSeconds);

            ui32PrevSeconds++;
        }

        /* Put the processor to sleep if there is nothing to do.  This allows
         * the CPU usage routine to measure the number of free CPU cycles.
         * If the processor is sleeping a lot, it can be hard to connect to
         * the target with the debugger. */
        SysCtlSleep();

        /* See if we have run int32_t enough and exit the loop if so. */
        if(ui32PrevSeconds > 10)
        {
            /* Indicate on the display that the example is stopped. */
            UARTprintf("\nStopped\n");

            /* Delete the task itself. */
            vTaskDelete(NULL);
        }
    }
}
/*-----------------------------------------------------------*/

static void prvInitApplication( void )
{
unsigned int uIdx;

    /* Header message for the demonstration. */
    UARTprintf("uDMA Memory to Memory Transfer Example\n");
    UARTprintf("\n");

    /* Show the clock frequency on the display. */
    UARTprintf("Tiva C Series @ %u MHz\n\n", configCPU_CLOCK_HZ / 1000000);

    /* Show statistics headings. */
    UARTprintf("CPU    Memory     Remaining\n");
    UARTprintf("Usage  Transfers  Time\n");

    /* Initialize the CPU usage measurement routine. */
    CPUUsageInit(configCPU_CLOCK_HZ, SYSTICKS_PER_SECOND, 2);

    /* Fill the source memory buffer with a simple incrementing pattern. */
    for(uIdx = 0; uIdx < MEM_BUFFER_SIZE; uIdx++)
    {
        g_ui32SrcBuf[uIdx] = uIdx;
    }
}
/*-----------------------------------------------------------*/

static void prvInitSWTransfer( void )
{
    /* Enable the uDMA controller at the system level.  Enable it to continue
     * to run while the processor is in sleep. */
    SysCtlPeripheralEnable(SYSCTL_PERIPH_UDMA);
    SysCtlPeripheralSleepEnable(SYSCTL_PERIPH_UDMA);

    /* Enable the uDMA controller error interrupt.  This interrupt will occur
     * if there is a bus error during a transfer. */
    IntEnable(INT_UDMAERR);

    /* Enable the uDMA controller. */
    uDMAEnable();

    /* Point at the control table to use for channel control structures. */
    uDMAControlBaseSet(ui8ControlTable);

    /* Enable interrupts from the uDMA software channel. */
    IntEnable(INT_UDMA);

    /* Put the attributes in a known state for the uDMA software channel.
     * These should already be disabled by default. */
    uDMAChannelAttributeDisable(UDMA_CHANNEL_SW,
                                UDMA_ATTR_USEBURST | UDMA_ATTR_ALTSELECT |
                                (UDMA_ATTR_HIGH_PRIORITY |
                                UDMA_ATTR_REQMASK));

    /* Configure the control parameters for the SW channel.  The SW channel
     * will be used to transfer between two memory buffers, 32 bits at a time.
     * Therefore the data size is 32 bits, and the address increment is 32 bits
     * for both source and destination.  The arbitration size will be set to 8,
     * which causes the uDMA controller to rearbitrate after 8 items are
     * transferred.  This keeps this channel from hogging the uDMA controller
     * once the transfer is started, and allows other channels cycles if they
     * are higher priority. */
    uDMAChannelControlSet(UDMA_CHANNEL_SW | UDMA_PRI_SELECT,
                          UDMA_SIZE_32 | UDMA_SRC_INC_32 | UDMA_DST_INC_32 |
                          UDMA_ARB_8);

    /* Set up the transfer parameters for the software channel.  This will
     * configure the transfer buffers and the transfer size.  Auto mode must be
     * used for software transfers. */
    uDMAChannelTransferSet(UDMA_CHANNEL_SW | UDMA_PRI_SELECT,
                           UDMA_MODE_AUTO, g_ui32SrcBuf, g_ui32DstBuf,
                           MEM_BUFFER_SIZE);

    /* Now the software channel is primed to start a transfer.  The channel
     * must be enabled.  For software based transfers, a request must be
     * issued.  After this, the uDMA memory transfer begins. */
    uDMAChannelEnable(UDMA_CHANNEL_SW);
    uDMAChannelRequest(UDMA_CHANNEL_SW);
}
/*-----------------------------------------------------------*/

void
uDMAErrorHandler(void)
{
    /* The interrupt handler for uDMA errors.  This interrupt will occur if the
     * uDMA encounters a bus error while trying to perform a transfer.  This
     * handler just increments a counter if an error occurs. */

uint32_t ui32Status;

    /* Check for uDMA error bit */
    ui32Status = uDMAErrorStatusGet();

    /* If there is a uDMA error, then clear the error and increment
     * the error counter. */
    if(ui32Status)
    {
        uDMAErrorStatusClear();
        g_ui32uDMAErrCount++;
    }
}
/*-----------------------------------------------------------*/

void
uDMAIntHandler(void)
{
    /* The interrupt handler for uDMA interrupts from the memory channel.  This
     * interrupt will increment a counter, and then restart another memory
     * transfer. */

uint32_t ui32Mode;

    /* Check for the primary control structure to indicate complete. */
    ui32Mode = uDMAChannelModeGet(UDMA_CHANNEL_SW);
    if(ui32Mode == UDMA_MODE_STOP)
    {
        /* Increment the count of completed transfers. */
        g_ui32MemXferCount++;

        /* Configure it for another transfer. */
        uDMAChannelTransferSet(UDMA_CHANNEL_SW, UDMA_MODE_AUTO,
                               g_ui32SrcBuf, g_ui32DstBuf, MEM_BUFFER_SIZE);

        /* Initiate another transfer. */
        uDMAChannelEnable(UDMA_CHANNEL_SW);
        uDMAChannelRequest(UDMA_CHANNEL_SW);
    }

    /* If the channel is not stopped, then something is wrong. */
    else
    {
        g_ui32BadISR++;
    }
}

